<?php

namespace App\Models;

use App\Helpers\Helper;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon ;
use Illuminate\Support\Facades\Validator;

use function PHPUnit\Framework\isEmpty;

class Coupon extends Model
{
    protected $table = 'coupons';

    private $lang ;

    // order is important for this array
    private $checks ;



    protected $fillable = [
        'name_ar',
        'name_en',
        'code',
        'start_date',
        'end_date',
        'max_use',
        'min_price',
        'discount',
        'discount_type',
        'type',
    ];


    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);

        $this->checks = [
            ['function' => 'isNotStart', 'fail_message' => __('home.This coupon is not valid yet.'), 'parameter' => []],
            ['function' => 'isNotEfficient', 'fail_message' => __('home.This coupon has expired.'), 'parameter' => []],
            ['function' => 'isUsed', 'fail_message' => __('home.You have already used this coupon.'), 'parameter' => []],
            ['function' => 'isMaxUsed', 'fail_message' => __('home.This coupon has reached its maximum usage limit.'), 'parameter' => []],
            ['function' => 'checkMinPrice', 'fail_message' => __('home.The total price does not meet the minimum required.'), 'parameter' => ['total_price']],
        ];
        // Set the current locale dynamically
        $this->lang = Helper::getLang();

    }

    public function delete()
    {
        $errors = [] ;
        if ($this->users()->exists()) {
            $errors[] = 'The used coupon cannot be deleted.';
        }

        if(count( $errors)){
            return $errors;
           }

           return parent::delete();
    }

    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'discount' => 'float',
    ];

    public function getNameAttribute(){
        return $this->{'name_'.$this->lang};
    }

    public function users()
    {
        return $this->hasMany(CouponUsed::class) ;
    }

    public function isMaxUsed()
    {
        return $this->users->count() >= $this->max_use ;
    }
    public function isNotEfficient()
    {
        return Carbon::parse($this->end_data)->startOfDay() < Carbon::now()->startOfDay();
    }


    public function isNotStart()
    {
        return  Carbon::parse($this->start_date)->startOfDay() > Carbon::now()->startOfDay();
    }

    public function isUsed()
    {
        return auth()->check()? $this->users->where('user_id',auth()->user()->id)->count() > 0 : false ;
    }

    public function checkMinPrice($total_price = 0): bool
    {

        return  $this->min_price > $total_price;
    }

    public function canUse($total_price = 0)
    {
        $can_use = true;
        $errors = [];

        //loop around check private paramter to validate counpon ;
        foreach ($this->checks as $check) {
            $function = $check['function'];
            $parameters = $check['parameter'];


            foreach ($parameters as &$parameter) {
                if ($parameter === 'total_price') {
                    $parameter = $total_price;
                }
            }
            // get function name from  curent item and called,
            // all those function return true or false
            $result = call_user_func_array([$this, $function], $parameters);


            if ($result) {
                // change status from and assign message currnt validarion to errors parm
                $can_use = false;
                $errors[] = $check['fail_message']; // Collect errors
            }
        }
        if (!$can_use) {
            // assign error message to globale errors parm ;
            Helper::messageError($errors) ;

        }
        return $can_use ;
    }




    public function getDiscountTypeTextAttribute()
    {
        return $this->discount_type ?  __('home.percentage') :__('home.amount') ;
    }


    public function getStartDateAttribute($value)
    {
        return  Carbon::parse($value) ->format('Y-m-d') ;
    }

    public function getEndDateAttribute($value)
    {
        return Carbon::parse($value) ->format('Y-m-d') ;
    }


}
