<?php

namespace App\Models;
use App\Models\OrderProduct;

use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    //
    protected $table = 'orders';
    protected $fillable = ['address_id','total_price','number' ,'user_id','shipping_id','products_price','coupon_id','payment_status','note','admin_seen','shipping_fees','coupon_discount','net'];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {
            $order->number = self::generateOrderNumber();
        });
    }

    public static function generateOrderNumber()
    {
        $latestOrder = self::orderBy('created_at', 'desc')->first();
        if ($latestOrder) {
            $lastOrderNumber = (int) substr($latestOrder->number, -6);
        } else {
            $lastOrderNumber = 0;
        }
        $newOrderNumber = str_pad($lastOrderNumber + 1, 6, '0', STR_PAD_LEFT);
        return config('site_prefix') . date('Ymd') . '-' . $newOrderNumber;
    }

    public function user(){
        return $this->belongsTo('App\Models\User','user_id');
    }

    public function address(){
        return $this->belongsTo(Address::class,'address_id');
    }

    public function getAddressAddressAttribute(){
        return $this->address?->address;
    }

    public function getAddressFullAttribute(){
        return $this->address?->full_address;
    }

    public function getAddressPhoneAttribute(){
        return $this->address?->phone;
    }


    public function products(){
        return $this->hasMany(OrderProduct::class);
    }

    public static function orderCount(){
	    return Order::where('admin_seen',0)->count();
	}
    public function orderStatus(){
       return $this->hasMany(OrderStatus::class);
    }

    public function status(){
        return $this->hasOne(OrderStatus::class)->latestOfMany();
    }

    public function getStatusCurrentAttribute(){
        return $this->status?->status;
    }
    public function getStatusCurrentNotAttribute(){
        return $this->status?->note;
    }
    public function delivery(){
        return $this->belongsTo(Delivery::class);
    }

    
    function scopeAdminFilter($query) {
        if(request('from')){
            $query->whereDate('created_at' ,'>=' ,request('from')) ;
        }
        if(request('to')){
            $query->whereDate('created_at' ,'<=' ,request('to')) ;
        }
        if(request('order_status')){
            $query->whereHas('status',function($q){
                $q->where('status', request('order_status')) ;
            } ) ;
        }
    }
}
